# n8n Workflow Patterns

## When to Use n8n

Use n8n when the agent needs to do anything beyond simple Cal.com booking:
- Update CRM (HubSpot, Salesforce, Airtable, GoHighLevel)
- Send SMS confirmations via Twilio
- Route to different calendars based on service type
- Multi-step workflows (book → CRM → SMS → Slack notification)

## Key Rules for Generated Workflows

1. Webhook node: `"httpMethod": "POST"`, `"responseMode": "responseNode"`
2. Always end with Respond to Webhook returning `{"result": "text agent speaks"}`
3. Always add error handler branch returning graceful fallback message
4. Use `{{ $json.args.field_name }}` to access parameters from Retell
5. Use placeholder credentials — tell user where to configure real ones
6. Generated JSON must be valid n8n import format with `nodes`, `connections`, and `settings`

## n8n Import Format

Every workflow JSON must follow this structure:

```json
{
  "name": "Workflow Name",
  "nodes": [...],
  "connections": {...},
  "settings": {
    "executionOrder": "v1"
  }
}
```

Each node needs: `id`, `name`, `type`, `position`, `parameters`, `typeVersion`

Connections map output pins to input pins:
```json
{
  "connections": {
    "Webhook": {
      "main": [[{"node": "Next Node Name", "type": "main", "index": 0}]]
    }
  }
}
```

## Pattern: Self-Selling Demo Lead Capture

For agency demo agents that capture prospect info and notify the owner.

```json
{
  "name": "Retell Self-Selling Demo",
  "nodes": [
    {
      "id": "1",
      "name": "Webhook",
      "type": "n8n-nodes-base.webhook",
      "position": [250, 300],
      "parameters": {
        "path": "demo-lead",
        "httpMethod": "POST",
        "responseMode": "responseNode"
      },
      "typeVersion": 2
    },
    {
      "id": "2",
      "name": "Add to Google Sheet",
      "type": "n8n-nodes-base.googleSheets",
      "position": [500, 200],
      "parameters": {
        "operation": "append",
        "documentId": "REPLACE_WITH_SHEET_ID",
        "sheetName": "Demo Leads",
        "columns": {
          "mappingMode": "defineBelow",
          "value": {
            "Date": "={{ new Date().toISOString() }}",
            "Business Name": "={{ $json.args.business_name }}",
            "Industry": "={{ $json.args.industry }}",
            "Contact Name": "={{ $json.args.contact_name }}",
            "Phone": "={{ $json.args.caller_phone }}",
            "Email": "={{ $json.args.caller_email }}",
            "Feedback": "={{ $json.args.demo_feedback }}",
            "Objections": "={{ $json.args.objections }}",
            "Outcome": "={{ $json.args.outcome }}",
            "Follow-up Date": "={{ $json.args.follow_up_date }}",
            "Call ID": "={{ $json.call_id }}"
          }
        }
      },
      "typeVersion": 4
    },
    {
      "id": "3",
      "name": "Check Outcome",
      "type": "n8n-nodes-base.if",
      "position": [750, 300],
      "parameters": {
        "conditions": {
          "string": [
            {
              "value1": "={{ $json.args.outcome }}",
              "operation": "equals",
              "value2": "closed"
            }
          ]
        }
      },
      "typeVersion": 2
    },
    {
      "id": "4",
      "name": "Notify Sale",
      "type": "n8n-nodes-base.gmail",
      "position": [1000, 150],
      "parameters": {
        "sendTo": "REPLACE_WITH_YOUR_EMAIL",
        "subject": "🎉 NEW SALE: {{ $('Webhook').item.json.args.business_name }}",
        "message": "You closed a sale on the demo line!\n\nBusiness: {{ $('Webhook').item.json.args.business_name }}\nIndustry: {{ $('Webhook').item.json.args.industry }}\nContact: {{ $('Webhook').item.json.args.contact_name }}\n\nFollow up to collect payment and start onboarding!"
      },
      "typeVersion": 2
    },
    {
      "id": "5",
      "name": "Notify Hot Lead",
      "type": "n8n-nodes-base.gmail",
      "position": [1000, 350],
      "parameters": {
        "sendTo": "REPLACE_WITH_YOUR_EMAIL",
        "subject": "🔥 Hot Lead: {{ $('Webhook').item.json.args.business_name }}",
        "message": "New demo lead!\n\nBusiness: {{ $('Webhook').item.json.args.business_name }}\nIndustry: {{ $('Webhook').item.json.args.industry }}\nOutcome: {{ $('Webhook').item.json.args.outcome }}\nFollow-up: {{ $('Webhook').item.json.args.follow_up_date }}\n\nCheck your Google Sheet for full details."
      },
      "typeVersion": 2
    },
    {
      "id": "6",
      "name": "Respond to Retell",
      "type": "n8n-nodes-base.respondToWebhook",
      "position": [1250, 300],
      "parameters": {
        "respondWith": "json",
        "responseBody": "={{ JSON.stringify({result: \"Got it! Thanks for checking out our demo.\"}) }}"
      },
      "typeVersion": 1
    }
  ],
  "connections": {
    "Webhook": {
      "main": [[{"node": "Add to Google Sheet", "type": "main", "index": 0}]]
    },
    "Add to Google Sheet": {
      "main": [[{"node": "Check Outcome", "type": "main", "index": 0}]]
    },
    "Check Outcome": {
      "main": [
        [{"node": "Notify Sale", "type": "main", "index": 0}],
        [{"node": "Notify Hot Lead", "type": "main", "index": 0}]
      ]
    },
    "Notify Sale": {
      "main": [[{"node": "Respond to Retell", "type": "main", "index": 0}]]
    },
    "Notify Hot Lead": {
      "main": [[{"node": "Respond to Retell", "type": "main", "index": 0}]]
    }
  },
  "settings": {"executionOrder": "v1"}
}
```

**Setup steps:**
1. Replace `REPLACE_WITH_SHEET_ID` with your Google Sheet ID
2. Replace `REPLACE_WITH_YOUR_EMAIL` with your email
3. Create sheet with columns: Date, Business Name, Industry, Contact Name, Phone, Email, Feedback, Objections, Outcome, Follow-up Date, Call ID
4. Connect Google Sheets and Gmail credentials

## Pattern: Lead Capture (Retell → Airtable + Slack)

```json
{
  "name": "Retell Lead Capture",
  "nodes": [
    {
      "id": "1",
      "name": "Webhook",
      "type": "n8n-nodes-base.webhook",
      "position": [250, 300],
      "parameters": {
        "path": "retell-lead",
        "httpMethod": "POST",
        "responseMode": "responseNode"
      },
      "typeVersion": 2
    },
    {
      "id": "2",
      "name": "Add to Airtable",
      "type": "n8n-nodes-base.airtable",
      "position": [500, 200],
      "parameters": {
        "operation": "create",
        "table": "Leads",
        "columns": {
          "mappingMode": "defineBelow",
          "value": {
            "Name": "={{ $json.args.caller_name }}",
            "Email": "={{ $json.args.caller_email }}",
            "Phone": "={{ $json.args.caller_phone }}",
            "Source": "Voice Agent",
            "Notes": "={{ $json.args.notes }}",
            "Status": "New"
          }
        }
      },
      "typeVersion": 2
    },
    {
      "id": "3",
      "name": "Notify Slack",
      "type": "n8n-nodes-base.slack",
      "position": [500, 400],
      "parameters": {
        "channel": "#leads",
        "text": "New lead from voice agent!\nName: {{ $json.args.caller_name }}\nPhone: {{ $json.args.caller_phone }}\nNotes: {{ $json.args.notes }}"
      },
      "typeVersion": 2
    },
    {
      "id": "4",
      "name": "Respond to Retell",
      "type": "n8n-nodes-base.respondToWebhook",
      "position": [750, 300],
      "parameters": {
        "respondWith": "json",
        "responseBody": "={{ JSON.stringify({result: \"Thank you! A member of our team will reach out to you within 24 hours.\"}) }}"
      },
      "typeVersion": 1
    }
  ],
  "connections": {
    "Webhook": {
      "main": [[
        {"node": "Add to Airtable", "type": "main", "index": 0},
        {"node": "Notify Slack", "type": "main", "index": 0}
      ]]
    },
    "Add to Airtable": {
      "main": [[{"node": "Respond to Retell", "type": "main", "index": 0}]]
    }
  },
  "settings": {"executionOrder": "v1"}
}
```

## Pattern: Booking + CRM + SMS

```json
{
  "name": "Retell Booking Handler",
  "nodes": [
    {
      "id": "1",
      "name": "Webhook",
      "type": "n8n-nodes-base.webhook",
      "position": [250, 300],
      "parameters": {
        "path": "retell-booking",
        "httpMethod": "POST",
        "responseMode": "responseNode"
      },
      "typeVersion": 2
    },
    {
      "id": "2",
      "name": "Route by Action",
      "type": "n8n-nodes-base.switch",
      "position": [500, 300],
      "parameters": {
        "dataType": "string",
        "value1": "={{ $json.args.action }}",
        "rules": {
          "rules": [
            {"value2": "book_appointment", "output": 0},
            {"value2": "capture_lead", "output": 1}
          ]
        }
      },
      "typeVersion": 3
    },
    {
      "id": "3",
      "name": "Book via Cal.com",
      "type": "n8n-nodes-base.httpRequest",
      "position": [750, 200],
      "parameters": {
        "url": "https://api.cal.com/v2/bookings",
        "method": "POST",
        "sendHeaders": true,
        "headerParameters": {
          "parameters": [
            {"name": "Authorization", "value": "Bearer cal_live_REPLACE"},
            {"name": "Content-Type", "value": "application/json"},
            {"name": "cal-api-version", "value": "2024-08-13"}
          ]
        },
        "sendBody": true,
        "bodyParameters": {
          "parameters": [
            {"name": "eventTypeId", "value": "=REPLACE_EVENT_TYPE_ID"},
            {"name": "start", "value": "={{ $json.args.preferred_time }}"},
            {"name": "attendee", "value": "={{ JSON.stringify({name: $json.args.caller_name, email: $json.args.caller_email, timeZone: 'America/Los_Angeles'}) }}"}
          ]
        }
      },
      "typeVersion": 4
    },
    {
      "id": "4",
      "name": "Send SMS",
      "type": "n8n-nodes-base.twilio",
      "position": [1000, 200],
      "parameters": {
        "operation": "send",
        "from": "=REPLACE_TWILIO_NUMBER",
        "to": "={{ $json.args.caller_phone }}",
        "message": "Hi {{ $json.args.caller_name.split(' ')[0] }}! Your appointment is confirmed. You'll receive a calendar invite shortly."
      },
      "typeVersion": 1
    },
    {
      "id": "5",
      "name": "Respond Success",
      "type": "n8n-nodes-base.respondToWebhook",
      "position": [1250, 200],
      "parameters": {
        "respondWith": "json",
        "responseBody": "={{ JSON.stringify({result: \"Your appointment is confirmed! You'll receive a confirmation text and calendar invite shortly.\"}) }}"
      },
      "typeVersion": 1
    },
    {
      "id": "6",
      "name": "Respond Lead Captured",
      "type": "n8n-nodes-base.respondToWebhook",
      "position": [750, 450],
      "parameters": {
        "respondWith": "json",
        "responseBody": "={{ JSON.stringify({result: \"I've noted your information. Our team will reach out within 24 hours.\"}) }}"
      },
      "typeVersion": 1
    }
  ],
  "connections": {
    "Webhook": {
      "main": [[{"node": "Route by Action", "type": "main", "index": 0}]]
    },
    "Route by Action": {
      "main": [
        [{"node": "Book via Cal.com", "type": "main", "index": 0}],
        [{"node": "Respond Lead Captured", "type": "main", "index": 0}]
      ]
    },
    "Book via Cal.com": {
      "main": [[{"node": "Send SMS", "type": "main", "index": 0}]]
    },
    "Send SMS": {
      "main": [[{"node": "Respond Success", "type": "main", "index": 0}]]
    }
  },
  "settings": {"executionOrder": "v1"}
}
```

## Retell Webhook Request Format

Retell sends to your webhook:
```json
{
  "args": {
    "caller_name": "John Doe",
    "caller_email": "john@example.com",
    "action": "book_appointment",
    ...other parameters defined in tool
  }
}
```

Access in n8n expressions: `{{ $json.args.caller_name }}`

## Response Format

Retell reads the `result` field and speaks it to the caller:
```json
{"result": "Your appointment is booked for Tuesday at 2pm."}
```

## Testing

1. In n8n: Set workflow to "Test" mode
2. Copy test URL (has `-test` in path)
3. Use curl or Postman to test:
```bash
curl -X POST "https://your-n8n.app.n8n.cloud/webhook-test/retell-booking" \
  -H "Content-Type: application/json" \
  -d '{"args":{"action":"book_appointment","caller_name":"Test User","caller_email":"test@test.com"}}'
```
4. Verify output, then switch to Production mode
5. Update Retell tool URL to production (remove `-test`)
